/**
\file unreal.fx

FX file for Unreal engine Direct3D 10 renderer.
Marijn Kentie 2009
*/

#include "unreal.fxh"

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
PS_INPUT VS( VS_INPUT input )
{
	PS_INPUT output = (PS_INPUT)0;

	output.origPos = input.pos;
	output.pos = input.pos;
	output.flags = input.flags;

	float4 projected = mul(output.pos,projection);
	if(useProjection)
	{
		output.pos = projected;
	}
	else //Handle tiles
	{		
		//Convert coordinates to [-1,1] range
		output.pos.x=-1+(2*output.pos.x*rViewportWidth);
		output.pos.y=-1+(2*output.pos.y*rViewportHeight);
		//Use perspective transformed depth for correct occlusion
		output.pos.z = projected.z/projected.w;
		output.pos.w = 1; //W set to 1 so X and Y aren't divided.
	}

	output.normal = input.normal;
	
	if(input.flags&PF_Modulated) //Modulated not influenced by color
	{
		output.color = float4(1,1,1,1);
	}
	else
	{
		output.color = clamp(input.color,0,1); //Color is sometimes >1 for Rune which screws up runestone particles
	}
		
	//From OpenGL renderer; seems fog should not be combined with these other effects
	if((input.flags & (PF_RenderFog|PF_Translucent|PF_Modulated|PF_AlphaBlend))==PF_RenderFog)
	{
		output.fog = input.fog*FOG_SCALE;
	}
	else
	{
		output.fog=float4(0,0,0,0);
	}
	
	for(int i=0;i<NUM_TEXTURE_PASSES;i++)
	{
		output.tex[i] = input.tex[i];
	}
	
	//d3d vs unreal coords
	output.pos.y =  -output.pos.y; 
	output.normal.y = -output.normal.y;
	
	
	//debug: to check if this flag is used
	if(input.flags & PF_Highlighted)
		output.color=float4(1,0,0,1);

	return output;
}


//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
PS_OUTPUT PS( PS_INPUT input)
{
	PS_OUTPUT output;
	//Initialize all textures to have no influence
	float4 color = input.color;
	float4 fog = input.fog;
	float4 diffuse = float4(1.0f,1.0f,1.0f,1.0f);
	float4 light = float4(1.0f,1.0f,1.0f,1.0f);
	float4 detail = float4(1.0f,1.0f,1.0f,1.0f);
	float4 fogmap = float4(0.0f,0.0f,0.0f,0.0f);
	float4 macro = float4(1.0f,1.0f,1.0f,1.0f);

	//Handle texture passes
	if(useTexturePass[0]) //Diffuse
	{
		//Alpha test
		if(input.flags&PF_Masked && !(input.flags&(PF_AlphaBlend|PF_Translucent)))
		{
			clip(textures[0].SampleBias(samPoint,input.tex[0],LODBIAS).a-0.5f);	
		}
		if(input.flags&PF_NoSmooth)
		{
			diffuse = textures[0].SampleBias(samPoint,input.tex[0],LODBIAS);
		}
		else
		{
			diffuse = textures[0].SampleBias(sam,input.tex[0],LODBIAS);
			if(input.flags&PF_AutoUPan || input.flags&PF_AutoVPan) //Sample skies a 2nd time for nice effect
			{
				diffuse = .5*diffuse+.5*textures[0].Sample(sam,input.tex[0]*2);
			}
		}
	}
	if(useTexturePass[1]) //Light
	{
		light = textures[1].SampleLevel(sam,input.tex[1],0);		
		light.xyzw = light.zyxw*2*LIGHT_SCALE; //Convert BGRA 7 bit to RGBA 8 bit	

	}
	if(useTexturePass[2]) //Detail (blend two detail texture samples with no detail for a nice effect).
	{
		//Interpolate between no detail and detail depending on how close the object is. Z=380 comes from UT D3D renderer.
		const int zFar = 380;
		float far = clamp(length(input.origPos),0,zFar)/zFar;
		detail = lerp(textures[2].SampleLevel(sam,input.tex[2],0),float4(1,1,1,1),far);
	}
	if(useTexturePass[3]) //Fog
	{		
		fogmap = textures[3].SampleLevel(sam,input.tex[3],0);				
		fogmap.xyzw = fogmap.zyxw*2*FOG_SCALE; //Convert BGRA 7 bit to RGBA 8 bit
	}
	if(useTexturePass[4]) //Macro
	{		
		macro = textures[4].SampleLevel(sam,input.tex[4],0);
	}
		
	output.color = color*diffuse*light*detail*macro+fogmap+fog;

	//Clear up flashes and brightness when implementing two pass rendering

	if(flashEnable && useProjection && !(input.flags&(PF_Translucent|PF_Modulated))) //Explosion, underwater, etc. effect; check projection to skip UI elements
	{
		output.color += flashColor;
	}

	//Apply brightness
	if(!(input.flags&PF_Modulated)) //Don't brighten DX mouse cursor, glasses, shadows, etc.
		output.color*=(.5+brightness);

	//Debug
//	if(input.pos.x<1)
	//	output.color=float4(1,0,0,1);
	return output;
}


//--------------------------------------------------------------------------------------
technique10 Render
{
	pass P0
	{
		SetVertexShader( CompileShader( vs_4_0, VS() ) );
		SetGeometryShader( NULL );
		SetPixelShader( CompileShader( ps_4_0, PS() ) );
		SetRasterizerState(rstate_Default);             
		
		//These states are (unfortunately) set by the DLL
		//SetDepthStencilState(dstate_Enable,1.0);
		//SetBlendState(bstate_NoBlend,float4(0,0,0,0),0xffffffff);
	}
}