/*##################################################################################################
##
##  BTCheckPoints 1.0
##  Copyright (C) 2010 Patrick "Sp0ngeb0b" Peltzer
##
##  This program is free software; you can redistribute and/or modify
##  it under the terms of the Open Unreal Mod License version 1.1.
##
##  Contact: spongebobut@yahoo.com | www.unrealriders.de
##
##
####################################################################################################
##
## Class: BTCheckPoints
##
## Description:
## BTCheckPoints Mutator
## ~~~~~~~~~~~~~~~~~~~~~
## BTCheckPoints allows players to set a checkpoint by saying '!cp' (or 'mutate checkpoint'), on
## which they will respawn after they die. Flag capping after using a Checkpoint is not possible.
## Checkpoints can be removed by saying '!nocp' (or 'mutate nocheckpoint').
## For better communication with the players, BTCheckPoints draws some usefull information on the
## HUD. This would be whether a player used his CP (and therefore cannot cap) and the name of the
## owner of a CP the player is touching.
## This mutator also features a cooperating mode to BTPlusPlus (only version 097r4 and higher, I
## recommend version 098 or higher), so that flag taking isn't even possible, using BTPlusPlus' flag
## disposer feature.
## BTCheckPoints also includes a Nexgen cooperating mode, which will automatically color its
## messages on the Nexgen Message HUD.
## Furthermore, BTCheckPoints is highly configurable (see below for more info).
##
##################################################################################################*/
class BTCheckPoints extends Mutator config(BTCheckPoints);

const version="1.0";

/*~~~~~~~~~~~~~~~~~
~ CONFIG SETTINGS ~
~~~~~~~~~~~~~~~~~~~*/

var config bool bEnabled;            // Whether we are enabled
var config bool bSearchForNexgen;    // Whether we should check for Nexgen Coop mode
var config bool bSearchForBTpp;      // Whether we should check for BTPlusPlus Coop mode
var config bool bDisallowCaps;       // Whether we should disallow caps after respawning at a CP
var config bool bRemoveCPAfterCap;   // Whether we should remove CPs after capping
var config int iCPDelay;             // Delay time in seconds before players can set a new CP
var config int iCPType;              // Type of the Checkpoint mesh (0=PlayerGhost, 1=Marking)
var config bool bAllowInAir;         // Whether players are allowed to set CPs while they are in air
var config bool bAllowInWater;       // Whether players can set a checkpoint while beeing underwater
var config bool bAllowInWrongZone;   // Whether players are allowed to set CPs while beeing in a special zone (e.g. lava or vaccum zone)
var config bool bAllowOnMovers;      // Whether players are allowed to set CPs while standing on a mover
var config bool bAllowOnTeleporters; // Whether players can set a CP while touching a teleporter. This fixes endless-dying-bugs on some servers.
var config bool bShowCPOwners;       // Whether players should get a message on their HUD informing about the CP owner if they are touching a CP
var config bool bShowCPStatus;       // Whether players should get a message on their HUD saying whether they used their CP


/*~~~~~~~~~~~
~ VARIABLES ~
~~~~~~~~~~~~*/

var bool Initialized;           // Whether we initialized

var string ServerPackages;      // String of the ServerPackages
var string ServerActors;        // String of the ServerActors

var bool Nexgen;                // Whether the server is running Nexgen
var string BTPlusPlus;          // Status of BTPlusPlus presence
var bool bCapPreventer;         // Whether we have to use our own Cap preventer if BTPlusPlus is not avaible

struct BTPlayers                // Structure of our BTPlayers
{
  	var PlayerPawn BTPP;        // The PlayerPawn
    var Checkpoint CP;          // Their Checkpoint (if they have one)
    var bool CPUsed;            // Whether the Player has respawned at his CP
    var float lastCPtime;       // Last time CP was set
    var int CurrTeam;           // Team number player is currently in
    var bool MessageSent;       // Whether we sent our welcome message
};
var BTPlayers BTP[32];

var PlayerPawn PlayerOwner;     // Client side owner





/*##################################################################################################
##
## replication -> Client replication
##
##################################################################################################*/
replication {

  // Replicate config vars to client
  reliable if (role == ROLE_Authority)
    bShowCPStatus, bShowCPOwners;

  reliable if (bShowCPStatus && role == ROLE_Authority) // Replicate structure to client if needed
    BTP;

}





/*##################################################################################################
# #################################################################################################
# #
# # GAME FUNCTIONS
# #
# #################################################################################################
##################################################################################################*/

/*##################################################################################################
##
## PreBeginPlay -> Register mutator and check for coop mutators
##
##################################################################################################*/
function PreBeginPlay() {

	if (Initialized) return;

  Initialized = True;
  
  // Create ini if doesn't exist.
	SaveConfig();
  
  if(!bEnabled) {
    Log("*");
   	Log("* BTCheckPoints: Version "$version$" by Sp0ngeb0b");
   	Log("*");
	  Log("* BTCheckPoints mutator NOT started, bEnabled is set to false!");
	  Log("* Unloading ..");
	  Log("*");
	  
	  destroy();
	  return;
  }

  // Get other mods running on server
  ServerPackages = CAPS(ConsoleCommand("get ini:Engine.Engine.GameEngine ServerPackages"));
  ServerActors = CAPS(ConsoleCommand("get ini:Engine.Engine.GameEngine ServerActors"));
  
  if(bSearchForNexgen) getNexgen();
  if(bSearchForBTpp) getBTPP();
  
  // Give out debugging log
  Log("*");
	Log("* BTCheckPoints mutator started");
	Log("* Version "$version$" by Sp0ngeb0b");
	
	if(!getBTCP()) Log("* WARNING: BTCheckPoints ServerPackage entry not found! This can cause unwanted results!");

	if(nexgen) Log("* Nexgen detected, cooperating mode active.");
	
	if(bSearchForBTpp) {
	  switch BTPlusPlus {
      case "ACTIVE":
        Log("* BTPlusPlus detected, cooperating mode active.");
      break;
      case "WRONG VERSION":
        Log("* Wrong BTPlusPlus version detected. Only version 97r4 or higher is supported.");
        bCapPreventer=true;
      break;
      case "NOT FOUND":
        Log("* BTPlusPlus NOT DETECTED.");
        bCapPreventer=true;
      break;
    }
  } else bCapPreventer=true;
  
  if(bDisallowCaps && bCapPreventer) Log("* Using own Cap-Preventer ...");
  
  if(bShowCPOwners || bShowCPStatus) {
    Log("* Starting client-side HUD.");
  }
  
	Log("* Contact: spongebobut@yahoo.com");
	Log("*");

	// Register mutator
	Level.Game.BaseMutator.AddMutator(self);
	Level.Game.RegisterMessageMutator(Self);
	
	// Set config var for our CPs
	class'Checkpoint'.default.CPType = iCPType;

}


/*##################################################################################################
##
## Tick -> Checks for left players, team-switchers and is capture preventer.
##         Also starts client-side HUD.
##
##################################################################################################*/
simulated function Tick(float DeltaTime) {
  local Pawn P;
  local PlayerPawn PP;
  local int i;
  
  Super.tick(DeltaTime);
  
  // Execute on client
  if( (bShowCPOwners || bShowCPStatus) && Level.NetMode != NM_DedicatedServer) {
    if(!bHUDMutator) RegisterHUDMutator();
  }
  
  searchForLeftPlayers();

  for(P=Level.PawnList; P!=None; P=P.NextPawn ) {
    if(P.isA('TournamentPlayer')) {
      
      PP = PlayerPawn(P);
      i = GetPlayerIndex(PP);
      
      if(Level.TimeSeconds - PP.PlayerReplicationInfo.StartTime >= 5 && !BTP[i].MessageSent) {
        BTP[i].MessageSent = true;
        
        PP.ClientMessage(message("BTCheckPoints version "$version$" by Sp0ngeb0b", "white"));
        PP.ClientMessage(message("Say '!cp' to add a Checkpoint, '!nocp' to remove it.", "white"));
      }
      
      // Fix for players beeing stuck after respawning in water
      if(bAllowInWater && PP.Region.Zone.bWaterZone && !PP.PlayerReplicationInfo.bWaitingPlayer && PP.health > 0
         && !Level.Game.bGameEnded && PP.Physics != PHYS_Swimming) {
        if (PP.HeadRegion.Zone.bWaterZone) PP.PainTime = PP.UnderWaterTime;
	   	  PP.setPhysics(PHYS_Swimming);
	  	  PP.GotoState('PlayerSwimming');
      }
      
      // Check whether player has changed his team
      if(BTP[i].CurrTeam != BTP[i].BTPP.PlayerReplicationInfo.Team) removeCheckpoint(PP, true);
      BTP[i].CurrTeam = BTP[i].BTPP.PlayerReplicationInfo.Team;

      if(BTP[i].CPUsed && bDisallowCaps) {

        // Cooperate with BTPlusPlus, so flags cant be picked up
        if(!bCapPreventer) PP.AttitudeToPlayer = ATTITUDE_Follow;
      
        // Prevent player from capping if he has flag
        if (PP.PlayerReplicationInfo.HasFlag != none) {
          CTFFlag(PP.PlayerReplicationInfo.HasFlag).SendHome();
				  PP.PlayerReplicationInfo.HasFlag = None;
        }
      }

    }
  }
}


/*##################################################################################################
##
## ModifyPlayer -> Handles player respawn
##
##################################################################################################*/
function ModifyPlayer(Pawn Other) {
  local PlayerPawn PP;
  local int i;

  if(!Other.IsA('TournamentPlayer')) {
    if (NextMutator != None) NextMutator.ModifyPlayer(Other);
    return;
  }
	
	PP = PlayerPawn(other);
  i = GetPlayerIndex(PP);


  if(BTP[i].CP != none && PP.PlayerReplicationInfo.deaths > 0) {  // Fixes ACE double ModifyPlayer bug

    Level.Game.PlayTeleportEffect(PP, false, true);

    PP.SetLocation(BTP[i].CP.orgLocation);
    PP.ClientSetRotation(BTP[i].CP.orgRotation);
    PP.viewRotation = BTP[i].CP.orgRotation;
    
    BTP[i].CPUsed = true;
    
  } else {
  
    BTP[i].CPUsed = false;
    if(!bDisallowCaps) PP.AttitudeToPlayer = ATTITUDE_Hate;
  }
  
  if (NextMutator != None) NextMutator.ModifyPlayer(Other);

}

/*##################################################################################################
##
## Mutate -> Handles player's mutate command
##
##################################################################################################*/
function Mutate(string MutateString, PlayerPawn Sender) {

  Super.Mutate(MutateString, Sender);

  switch MutateString {
    case "checkpoint":
      SetCheckpoint(Sender);
    break;
    case "nocheckpoint":
      removeCheckpoint(Sender, false);
    break;
  }
}


/*##################################################################################################
##
## MutatorTeamMessage -> Handles players say messages
##
##################################################################################################*/
function bool MutatorTeamMessage(Actor Sender, Pawn Receiver,
                                 PlayerReplicationInfo PRI, coerce string S, name Type,
                                 optional bool bBeep) {

	if(Sender.IsA('TournamentPlayer') && Sender == Receiver) {
		if(S ~= "!cp" || S ~= "!checkpoint") {
      SetCheckpoint(PlayerPawn(Sender));
    } else if(S ~= "!nocp" || S ~= "!nocheckpoint") {
      removeCheckpoint(PlayerPawn(Sender), false);
    }
	}
	
	// Allow other message mutators to do their job.
  if (nextMessageMutator != none) {
    return nextMessageMutator.mutatorTeamMessage(sender, receiver, pri, s, type, bBeep);
  } else {
    return true;
  }
}


/*##################################################################################################
##
## MutatorBroadcastLocalizedMessage -> Removes CPs after capping
##
##################################################################################################*/
function bool MutatorBroadcastLocalizedMessage(Actor Sender, Pawn Receiver, out class<LocalMessage>
                                               Message, out optional int Switch,
                                               out optional PlayerReplicationInfo RelatedPRI_1,
                                               out optional PlayerReplicationInfo RelatedPRI_2,
                                               out optional Object OptionalObject) {
  local PlayerPawn PP;
  local int i;
                                               
  if(!bremoveCPAfterCap || Message != class'CTFMessage' ||
     !Sender.IsA('CTFGame')) return super.MutatorBroadcastLocalizedMessage(Sender, Receiver, Message, Switch, RelatedPRI_1, RelatedPRI_2, OptionalObject);

  // Only call this once (not on all players), so check if the Receiver is the capturer.
  if(Switch == 0 && Receiver == Pawn(RelatedPRI_1.Owner)) {
  
    PP = PlayerPawn(Receiver);
    i = GetPlayerIndex(PP);
    
    if(BTP[i].CP != none) removeCheckpoint(PP, true);
  }
  
  return super.MutatorBroadcastLocalizedMessage(Sender, Receiver, Message, Switch, RelatedPRI_1, RelatedPRI_2, OptionalObject);
}
        

/*##################################################################################################
##
## AddMutator -> Protection against initializing this mutator twice.
##
##################################################################################################*/
function AddMutator(Mutator M)
{
  if(M == none) return;
  
	if (M.Class != Class) Super.AddMutator(M);
	else if(M != Self) M.Destroy();
}










/*##################################################################################################
# #################################################################################################
# #
# # HUD FUNCTIONS
# #
# #################################################################################################
##################################################################################################*/

/*##################################################################################################
##
## PostRender -> Used for drawing on the player's HUD
##
##################################################################################################*/
simulated event PostRender(canvas Canvas) {
  local CheckPoint CP;

  // Get PlayerOwner.
  if( PlayerOwner == None ) {
    PlayerOwner = Canvas.Viewport.Actor;
  }
  
  if(bShowCPStatus) DrawCPStatus(Canvas);
  
  if(bShowCPOwners) {
  
    foreach VisibleCollidingActors(class'Checkpoint', CP, 64.0, PlayerOwner.Location) {
      if(CP.owner != none && PlayerPawn(CP.owner) != PlayerOwner) {
        DrawCPName(Canvas, PlayerPawn(CP.owner).PlayerReplicationInfo.PlayerName);
        break;
      }
    }

  }
  
  // Always call next HUD mutator to keep the chain
  if(NextHUDMutator != none) NextHUDMutator.PostRender(Canvas);
}


/*##################################################################################################
##
## DrawCPStatus -> Draws on the HUD whether the player has used his CP (so capping wont be possible)
##                 Used code of BTPlusPlus for better integration
##
##################################################################################################*/
simulated function DrawCPStatus(Canvas Canvas) {
  local int i;
  local float W, H;
  
  if(PlayerOwner.PlayerReplicationInfo.bIsSpectator || PlayerOwner.PlayerReplicationInfo.bWaitingPlayer) return;

  Canvas.Reset();
	Canvas.Font = Canvas.SmallFont;
	Canvas.TextSize("CHECKPOINT USED: ", W, H);
	Canvas.SetPos(5, Canvas.ClipY/2-4.5*H-1);
	Canvas.DrawColor = ChallengeHUD(PlayerOwner.myHUD).WhiteColor;
  DrawShadowText(Canvas, "CHECKPOINT USED: ");
	Canvas.SetPos(5+W, Canvas.ClipY/2-4.5*H-1);
	
  i = GetPlayerIndex(PlayerOwner);

  if(BTP[i].CPUsed) {
		Canvas.DrawColor = ChallengeHUD(PlayerOwner.myHUD).RedColor;
		DrawShadowText(Canvas, "YES");
	}	else {
		Canvas.DrawColor = ChallengeHUD(PlayerOwner.myHUD).GreenColor;
		DrawShadowText(Canvas, "NO");
  }
	
}


/*##################################################################################################
##
## DrawShadowText -> Used code of BTPlusPlus for better integration
##
##################################################################################################*/
simulated function DrawShadowText (Canvas Canvas, coerce string Text, optional bool Param) {
  local Color OldColor;
  local int XL,YL;
  local float X, Y;

  OldColor = Canvas.DrawColor;

  Canvas.DrawColor.R = 0;
  Canvas.DrawColor.G = 0;
  Canvas.DrawColor.B = 0;

	XL = 1;
	YL = 1;
	X=Canvas.CurX;
	Y=Canvas.CurY;
	Canvas.SetPos(X+XL,Y+YL);
	Canvas.DrawText(Text, Param);
	Canvas.DrawColor = OldColor;
	Canvas.SetPos(X,Y);
	Canvas.DrawText(Text, Param);
}


/*##################################################################################################
##
## DrawCPName -> Draws the name of a checkpoint-owner
##
##################################################################################################*/
simulated function DrawCPName( canvas Canvas, string ValueString ) {
	local float XL, YL, XOffset, X1;

	Canvas.Style = ChallengeHUD(PlayerOwner.MyHUD).Style;
	Canvas.Font = ChallengeHUD(PlayerOwner.MyHUD).MyFonts.GetMediumFont(Canvas.ClipX);
	
	Canvas.StrLen("Checkpoint of ", XL, YL);
	X1 = XL;
	Canvas.StrLen(ValueString, XL, YL);
	XOffset = Canvas.ClipX/2 - (X1+XL)/2;
	Canvas.SetPos(XOffset, Canvas.ClipY - 192 * ChallengeHUD(PlayerOwner.MyHUD).Scale);
	XOffset += X1;
	
	Canvas.DrawColor = ChallengeHUD(PlayerOwner.MyHUD).WhiteColor;
	
	Canvas.DrawText("Checkpoint of");
	
	Canvas.SetPos(XOffset, Canvas.ClipY - 192 * ChallengeHUD(PlayerOwner.MyHUD).Scale);
	Canvas.DrawText(ValueString);
}










/*##################################################################################################
# #################################################################################################
# #
# # OWN FUNCTIONS
# #
# #################################################################################################
##################################################################################################*/

/*##################################################################################################
##
## setCheckpoint -> Sets a new checkpoint for a specific Player.
##
##################################################################################################*/
function setCheckpoint(PlayerPawn Sender) {
  local int i;
  local Teleporter TP;

  // Make sure we have a real player
  if(Sender != none && !Sender.IsA('TournamentPlayer')) return;

  i = GetPlayerIndex(Sender);

  // Check whether the player is allowed to set a checkpoint
  if(Sender.PlayerReplicationInfo.bWaitingPlayer || Level.Game.bGameEnded) {
    Sender.ClientMessage(message("You cannot set a Checkpoint at this time.", "red"));
    return;
  }

  if(Sender.health < 1) {
    Sender.ClientMessage(message("You need to be alive to set a Checkpoint.", "red"));
    return;
  }

  if(!bAllowInAir && Sender.Velocity.Z != 0 && !Sender.Region.Zone.bWaterZone) {
    Sender.ClientMessage(message("You can only set a Checkpoint while standing on the ground.", "red"));
    return;
  }

  if(!bAllowInWrongZone) {
    if(Sender.Region.Zone != none && ( (!bAllowInWater && Sender.Region.Zone.bWaterZone) || Sender.Region.Zone.bPainZone ||
       Sender.Region.Zone.bKillZone || Sender.Region.Zone.isA('WarpZone') ||
       Sender.Region.Zone.isA('TeleporterZone') || Sender.Region.Zone.isA('VacuumZone')) ) {
      Sender.ClientMessage(message("You cannot set a Checkpoint in this zone.", "red"));
      return;
    }
  }

  if(!bAllowOnMovers && Sender.Base != none && Sender.Base.isA('Mover')) {
    Sender.ClientMessage(message("You cannot set a Checkpoint while standing on a mover.", "red"));
    return;
  }
  
  // Fixes endless-dying bug which occurs on some servers
  if(!bAllowOnTeleporters) {
  
    foreach RadiusActors(class'Teleporter', TP, 24.0, Sender.Location) {
      Sender.ClientMessage(message("You cannot set a Checkpoint at this position.", "red"));
      return;
    }
  }

  if( iCPDelay > 0 && iCPDelay >= int(level.timeSeconds - BTP[i].lastCPtime)) {
    Sender.ClientMessage(message("You have to wait "$getDelayTime(level.timeSeconds, BTP[i].lastCPtime)$" until you can set a new Checkpoint.", "red"));
    return;
  }

  // Remove old checkpoint
  if(BTP[i].CP != none) {
    BTP[i].CP.destroy();
    BTP[i].CP = none;
  }

  // Spawn new checkpoint at player's location
  BTP[i].CP = spawn(class'Checkpoint', Sender, , Sender.Location, Sender.Rotation);

  // Update last Checkpoint time
  BTP[i].lastCPtime = level.timeSeconds;

  Sender.ClientMessage(message("Checkpoint set.", "green"));
}


/*##################################################################################################
##
## removeCheckpoint -> Removes the checkpoint of a specific Player.
##
##################################################################################################*/
function removeCheckpoint(PlayerPawn Sender, bool bForced) {
  local int i;

  // Make sure we have a real player
  if(!Sender.IsA('TournamentPlayer')) return;

  i = GetPlayerIndex(Sender);

  if(BTP[i].CP != none) {
    BTP[i].CP.destroy();
    BTP[i].CP = none;
    if(!bforced) Sender.ClientMessage(message("Checkpoint deleted.", "green"));
    else Sender.ClientMessage(message("Checkpoint removed.", "red"));
  }
}


/*##################################################################################################
##
## searchForLeftPlayers -> Removes old CPs and clears data
##
##################################################################################################*/
function searchForLeftPlayers() {
  local int i;

  // Search existing entries
  for (i=0;i<ArrayCount(BTP);i++) {
    if (BTP[i].BTPP == none || BTP[i].BTPP.Player == none) {
      if(BTP[i].CP != none) BTP[i].CP.destroy();

      BTP[i].CP = none;
      BTP[i].CPUsed = false;
      BTP[i].lastCPtime = 0;
      BTP[i].MessageSent = false;
    }
  }
}


/*##################################################################################################
##
## GetPlayerIndex -> Retrieves the playerpawn's ID in the structure or creates a new entry
##
##################################################################################################*/
simulated function int GetPlayerIndex(PlayerPawn Player) {
  local int i, FirstEmptySlot;

  FirstEmptySlot = -1;

  // Search existing entries
  for (i = 0; i < ArrayCount(BTP); i++) {
    if (BTP[i].BTPP == Player) break;
    else if (BTP[i].BTPP == None && FirstEmptySlot == -1) FirstEmptySlot = i;
  }

  // Not found, create new entry
  if (i == ArrayCount(BTP)) {
    i = FirstEmptySlot;
    BTP[i].BTPP = Player;
    BTP[i].CP = none;
    BTP[i].CPused = false;
    BTP[i].MessageSent = false;
  }
  return i;
}


/*##################################################################################################
##
## message -> Adds Nexgen message HUD color-strings to the message
##
##################################################################################################*/
function string message(string message, string color) {

  // Check if Nexgen is avaible
  if(!nexgen) return message;

  switch color {
    case "red": return "<C00>"$message;
    break;
    case "white": return "<C04>"$message;
    break;
    case "green": return "<C02>"$message;
    break;
    default: return message;
  }
}


/*##################################################################################################
##
## getBTCP -> Checks whether the BTCheckpoint ServerPackage has been added
##
##################################################################################################*/
function bool getBTCP() {

	if(InStr(ServerPackages,"BTCHECKPOINTS") != -1) return true;
	else return false;
}

/*##################################################################################################
##
## getNexgen -> Checks whether Nexgen is running on server
##
##################################################################################################*/
function getNexgen() {

	if(InStr(ServerActors,"NEXGEN") != -1) nexgen=true;
}


/*##################################################################################################
##
## getBTPP -> Checks whether BTPlusPlus is running on server
##
##################################################################################################*/
function getBTPP() {
  local actor a;

	foreach Level.AllActors(class 'actor',a) {
		if(left(a.Class, 10) == "BTPlusPlus") {
      if(getBTPPv()) {

		  	BTPlusPlus="ACTIVE";
			  break;
      } else {
        BTPlusPlus="WRONG VERSION";
        break;
	  	}
	 }
 }

  if(BTPlusPlus == "") BTPlusPlus="NOT FOUND";
}


/*##################################################################################################
##
## getBTPPv -> Checks whether there is a compatible BTPlusPlus version running on server
##
## Compatible versions: Everything starting at v097r4
##
##################################################################################################*/
function bool getBTPPv() {

	if(int(Mid(Serverpackages, InStr(ServerPackages,"BTPLUSPLUSV") + 11, 3)) >= 098) return true;
	else if(int(Mid(Serverpackages, InStr(ServerPackages,"BTPLUSPLUSV097R") + 15, 1)) >= 4) return true;
	else return false;
}


/*##################################################################################################
##
## getDelayTime -> Retrieves delay until players can set new cps in secs
##
##################################################################################################*/
function string getDelayTime(float high, float low) {
  local int delay;

  delay = int(iCPDelay-(high-low)) + 1;

  if(delay == 1) return "1 second";
  else return delay$" seconds";
}




/*##################################################################################################
# #################################################################################################
# #
# # DEFAULTPROPERTIES
# #
# #################################################################################################
##################################################################################################*/
defaultproperties
{
  bEnabled=True
  bSearchForNexgen=True
  bSearchForBTpp=True
  bDisallowCaps=True
  bRemoveCPAfterCap=True
  iCPDelay=5
  iCPType=0
  bAllowInAir=False
  bAllowInWater=True
  bAllowInWrongZone=False
  bAllowOnMovers=False
  bAllowOnTeleporters=False
  bAlwaysTick=True
  bAlwaysRelevant=True
  RemoteRole=ROLE_SimulatedProxy
}


