/*
    BTPlusPlus 0.97
    Copyright (C) 2004-2006 Damian "Rush" Kaczmarek

    This program is free software; you can redistribute and/or modify
    it under the terms of the Open Unreal Mod License version 1.1.
*/

class BTPlusPlus extends Mutator config(BTPlusPlus);

var const string Version;

var bool bInitialized;
var int CurrentID;
var color GreenColor;
var color RedColor;

var string LevelName; // current level name

var bool bSpawnFlags; // used to tell the Timer() function what to do
var bool bFlagsSpawned; // used to control the Timer() behaviour

// Here are the config variables
var config bool bEnabled;
var config bool bBTScoreboard;
var config bool bAutoLoadInsta;
var config bool bMultiFlags;
var config bool bRespawnAfterCap;
var config bool bAntiBoost;
var config bool bBlockBoostForGood;
var config string AllowBoostOn;
var config bool bNoKilling;
var config string AllowKillingOn;
var config bool bGhostWhenCamping;
var config int CampTime;
var config int CampRadius;
var config bool bEverybodyGhosts;
var config bool bForceMoversKill;
var config string ForceMoversKill[10];
var config bool bSaveRecords;
var config bool bNoCapSuicide;
var config int RecordsWithoutBoost; // 0 - disabled, 1 -  enabled, 2 - enabled without cooperation maps
var config bool bDisableInTournament;
var config bool bDisableInNonBTMaps;
var config string BoardLabel;
var config string CountryFlagsPackage;

struct PlayerInfo
{
	var 		Pawn Player;
	var int 	PlayerID;
	var vector  LastPlayerLoc;
	var         BTPPReplicationInfo RI;
	var			UserConfig Config;
	var bool	bHelpSent;
	var int		BestTime;
};
var PlayerInfo PI[32];

struct SpecInfo
{
	var	Pawn Spec;
	var BTPPReplicationInfo RI;
};
var SpecInfo SI[32];

var float cTime; // for anti-camper timer in tick

var bool bAllowKilling; // can be different than bNoKilling because of AllowKillingOn
var bool bAllowBoost;  // only important if bBlockBoostForGood

var CTFFlag TempFlagList[4]; // used to save the old flag icons and restore them if needed

var int MapBestTime;
var string MapBestPlayer;

// BTPlusPlus other objects
var BTPPHUDNotify HUD;
var RecordData RD;
var Mutator Insta;
var BTPPGameReplicationInfo GRI;

var UBrowserHTTPClient IpToCountry; // IpToCountry external actor for resolving country names
var bool bNoIpToCountry; // will be true if CountryFlags texture is not in the serverpackages

var bool bCooperationMap;

// actors to which the custom events will be send
var Actor EventHandlers[10];
var int EventHandlersCount;

//====================================
// CheckDisableNeed - Check whether to run and if to use IpToCountry module
// Triggered in: PreBeginPlay
//====================================
function int CheckDisableNeed()
{
	local string packages;
	local BTPlusPlus temp;

	packages=ConsoleCommand("get ini:Engine.Engine.GameEngine ServerPackages");

	if(InStr(Caps(packages), Caps(CountryFlagsPackage)) == -1)
		bNoIpToCountry=True;
	if(!bEnabled)
		return 1;
	if(InStr(Caps(packages), Caps("BTPlusPlusv0"$Version$"_C")) == -1) // check if we're in the serverpackages
		return 2;
	if(!ClassIsChildOf(Level.Game.class, class'CTFGame')) // check for a CTF game
		return 3;
	if(bDisableInTournament && class<DeathMatchPlus>(Level.Game.Class).Default.bTournament) // check for Tournament running
		return 4;
	if(bDisableInNonBTMaps && Caps(Left(string(Level), 3))!="BT-" && Caps(Left(string(Level), 7))!="CTF-BT-" && Caps(Left(string(Level), 3))!="BT+" && Caps(Left(string(Level), 7))!="CTF-BT+") // check for a BT or BT+ map
		return 5;
	foreach AllActors(class'BTPlusPlus', temp) // check if there isn't another instance running
		if(temp != self)
			return 6;
	return 0;
}

//====================================
// PreBeginPlay - Mutator registration, Auto loading instagib, Spawning classes, Setting scoreboard
//====================================
function PreBeginPlay()
{
	if(bInitialized)
		return;
	bInitialized=True;

	Tag='BTPlusPlus';

	log("+-----------------");
	log("| BTPlusPlus v0."$Version$" by [es]Rush*bR");
	switch (CheckDisableNeed())
	{
		case 1:
			log("| Status: Disabled");
			log("| Reason: bEnabled=False in the config");
			log("+-----------------");
			Destroy();
			return;
		case 2:
			log("| Status: Disabled");
			log("| Reason: BTPlusPlusv0"$Version$"_C is not in ServerPackages!");
			log("+-----------------");
			Destroy();
			return;
		case 3:
			log("| Status: Disabled");
			log("| Reason: Gametype is not based on CTF!");
			log("+-----------------");
			Destroy();
			break;
			return;
		case 4:
			log("| Status: Disabled");
			log("| Reason: Server is in Tournament mode and bDisableInTournament=True");
			log("+-----------------");
			Destroy();
			return;
		case 5:
			log("| Status: Disabled");
			log("| Reason: Map is not a BT map and bDisableInNonBTMaps=True");
			log("+-----------------");
			Destroy();
			return;
		case 6:
			log("| Status: Disabled");
			log("| Reason Another instance of BTPlusPlus detected!");
			log("+-----------------");
			Destroy();
			return;
		case 0:
			log("| Status: Running");
			log("+-----------------");
	}

	Level.Game.BaseMutator.AddMutator( self );
	Level.Game.RegisterDamageMutator( Self );
	Level.Game.RegisterMessageMutator(Self);
	if(bAutoLoadInsta)
	{
		if((!bDisableInTournament || !class<DeathMatchPlus>(Level.Game.Class).Default.bTournament)
		&& (!bDisableInNonBTMaps || (Left(string(Level), 3)=="BT-" || Left(string(Level), 7)=="CTF-BT-")))
		{
			RemoveActorsForInsta();
			Insta=Level.spawn(class'InstaGibBT');
			Insta.DefaultWeapon=class'SuperShockRifleBT';
			Level.Game.BaseMutator.AddMutator(Insta);
		}
	}
	GRI = Level.spawn(class'BTPPGameReplicationInfo');
	HUD = Level.spawn(class'BTPPHUDNotify');
	GRI.bShowAntiBoostStatus = bAntiBoost;
	if(bAntiBoost)
		GRI.bShowAntiBoostStatus = !bBlockBoostForGood;
	GRI.bSaveRecords = bSaveRecords;
	GRI.CountryFlagsPackage = CountryFlagsPackage;
	GRI.BoardLabel = BoardLabel;
	if(bBTScoreboard)
		Level.Game.ScoreBoardType=class'BTScoreBoard';
}

//====================================
// PreBeginPlay - initializing RecordData, Binding IpToCountry, Retrieving saved records, Setting timer to spawn custom flags, Setting movers to kill, Setting killing block
//====================================
function PostBeginPlay()
{
	local int i;
	local bool bTournament, bNonBTMap;
	local string temp;
	local int temp2;
	local mutator Mut;

	LevelName = GetLevelName();

	// these are the symbols of cooperation maps, for 3 players or 2 players
	if(InStr(LevelName, "-III") != -1 || InStr(LevelName, "-II") != -1)
		bCooperationMap=True;

	RD = spawn(class'RecordData');
	RD.Main = Self;

	if(!bNoIpToCountry)
		foreach AllActors(class'UBrowserHTTPClient', IpToCountry)
		{
			if(string(IpToCountry.class) == "IpToCountry.LinkActor")
				break;
			else
				IpToCountry=None;
		}
	if(RD.RecordsNum == RD.default.RecordsNum && RD.Record[0] != "") // RecordsNum is a var intoduced in 097 version and thus if it is set to default, either that's a new config or a config from a previous version
		RD.ConvertToNewFormat();
	if(bSaveRecords)
	{
		temp = RD.CheckRecord(LevelName);
		temp2 = int(SepLeft(temp, chr(9)));
		if(temp2 > 0 && temp2 < 2000)
		{
			MapBestTime = temp2;
			MapBestPlayer = DelSpaces(SelElem(temp, 2, chr(9)));
			GRI.MapBestDate = SelElem(temp, 3, chr(9));
			GRI.MapBestTime = FormatScore(MapBestTime);
			GRI.MapBestPlayer = MapBestPlayer;
		}
	}
	//Create ini if doesn't exist.
	SaveConfig();

	cTime = 0.0;

	bSpawnFlags=true;
		// flags aren't here yet so we have to mess with them a moment later ...
		// flags still have to be original after 1 second after the game start cause there is a custom flag mutator on the market which replaces them
	SetTimer(1.1, false);
	bFlagsSpawned=True;

	if(bForceMoversKill)
			CheckMovers();

	if(bNoKilling)
		CheckBlockKilling();
	if(bAllowBoost)
		CheckAllowBoost();

	SendEvent("btpp_started");
}

//====================================
// Tick - Camper detection, New player detection
// Inherited from class'Actor'
//====================================
function tick(float DeltaTime)
{
	local int i;
	Super.tick(DeltaTime);

	// Camper detection
	if(bGhostWhenCamping && !bEverybodyGhosts)
	{
		cTime += DeltaTime;
		if (cTime > CampTime)
		{
			CheckPositions();
			cTime = 0.0;
		}
	}
	CheckForNewPlayer();
}

//====================================
// Timer - Setting flags collision, Clearing flag list, Sending help
// Triggered in: PostBeginPlay, ModifyPlayer
//====================================
function Timer()
{
	local Pawn P;
	local FlagBase FB;
	local FlagDisposer FlagD;
	local CTFFlag TmpFlag;
	local CTFFlag Flag[2];
	local int i;

	if(bSpawnFlags)
	{
		Foreach AllActors(Class'CTFFlag',TmpFlag)
		{
			if(TmpFlag.Team>1)
				continue;
			Flag[TmpFlag.Team]=TmpFlag; // save the flags if we ever w
			// makes the flag untouchable(disables Touch function in it)
			TmpFlag.SetCollision(false, false, false);
		}
		for(i=0;i<4;i++)
		{
			// Save the flag list for future restore
			TempFlagList[i]=CTFReplicationInfo(Level.Game.GameReplicationInfo).FlagList[i];
			// It makes the flag icons vanish on the player hud so that we can draw our own
			CTFReplicationInfo(Level.Game.GameReplicationInfo).FlagList[i]=None;
		}

	// spawn the custom flag scripts on place of original
		Foreach AllActors(Class'FlagBase',FB)
		{
			if(FB.Team == 1)
			{
				FlagD=Spawn(class'FlagDisposer',,, FB.Location);
				FlagD.Controller = Self;
				FlagD.Homebase = FB;
				FlagD.Team = 1;
				FlagD.Flag = Flag[1];
			}
			else
			{
				FlagD=Spawn(class'FlagDisposer',,, FB.Location);
				FlagD.Controller = Self;
				FlagD.Homebase = FB;
				FlagD.Team = 0;
				FlagD.Flag = Flag[0];
			}
		}
		if(!bMultiFlags)
			SetMultiFlags(false);
		bSpawnFlags=False;
		return;
	}

	for( i=0;i<32;i++ )
	{
		if(PI[i].RI == none)
			continue;
		if(PI[i].bHelpSent || PI[i].Player.PlayerReplicationInfo.bWaitingPlayer)
		{
			continue;
		}
		SendHelp(PI[i].Player);
		PI[i].bHelpSent=True;
	}
}

//====================================
// CheckMovers - Set predefined movers to kill players or set the default values
// Triggered in: PostBeginPlay, Mutate
//====================================
function CheckMovers(optional bool bDefaults)
{
	local int i, j;
	local bool bMapMatch;
	local mover M;
	local Actor A;

	if(bDefaults)
	{
		Foreach AllActors(Class'Mover',M)
		{
			M.MoverEncroachType=M.default.MoverEncroachType;
		}
	}
	for(i=0;i<10;i++)
	{
		if(ForceMoversKill[i]=="")
			continue;
		for(j=1;j<=ElementsNum(SepRight(ForceMoversKill[i]));j++)
		{
			if(InStr(Caps(LevelName), Caps(SelElem(SepRight(ForceMoversKill[i]),j, ","))) != -1)
			{
				bMapMatch=True;
				break;
			}
		}
		if(bMapMatch)
		{
			for(j=1;j<=ElementsNum(SepLeft(ForceMoversKill[i]));j++)
			{
				Foreach AllActors(Class'Mover',M)
				{
					if(string(M.Name)==SelElem(SepLeft(ForceMoversKill[i]), j, ","))
						M.MoverEncroachType=ME_CrushWhenEncroach;
				}
			}
		}
	}
}

//====================================
// CheckBlockKilling - On predefined maps allow killing
// Triggered in: PostBeginPlay, Mutate
//====================================
function bool CheckBlockKilling()
{
	local int i;
	local string elem;

	for(i=1;i<=ElementsNum(AllowKillingOn);i++)
	{
		elem=SelElem(AllowKillingOn, i, ",");
		if(elem == "")
			return false;
		if(InStr(Caps(LevelName), Caps(elem)) != -1)
		{
			bAllowKilling=True;
			return true;
			break;
		}
	}
	bAllowKilling=False;
	return false;
}

//====================================
// CheckAllowBoost - On predefined maps allow boosting if it is normally forbidden
// Triggered in: PostBeginPlay, Mutate
//====================================
function bool CheckAllowBoost()
{
	local int i;
	local string elem;

	for(i=1;i<=ElementsNum(AllowBoostOn);i++)
	{
		elem=SelElem(AllowBoostOn, i, ",");
		if(elem == "")
			return false;
		if(InStr(Caps(LevelName), Caps(elem)) != -1)
		{
			bAllowBoost=True;
			return true;
			break;
		}
	}
	bAllowBoost=False;
	return false;
}

//#########################################################################
//### ANTI-CAMPER FUNCTIONS
//#########################################################################
//====================================
// CheckPositions - Check player for camping
// Triggered in: Tick
//====================================
function CheckPositions()
{
	local int i;
	local float VDistance;
	local Vector DVector;

	for(i=0;i<32;i++)
	{
		if(PI[i].Player == none)
			continue;
		DVector = PI[i].Player.Location - PI[i].LastPlayerLoc;
		VDistance = VSize(DVector);
		if (VDistance < CampRadius)
			SetGhost(PI[i].Player,true);
		else
			SetGhost(PI[i].Player,false);
		PI[i].LastPlayerLoc = PI[i].Player.Location;
	}
}

//====================================
// SetGhost - Change player's look to be like a ghost
// Triggered in: CheckPositions, EnableAllGhost, DisableAllGhost
//====================================
function SetGhost(Pawn P, bool Status)
{
	if(Status)
	{
		P.bBlockPlayers = false;
		P.SetDisplayProperties(STY_Translucent, P.default.Texture, P.default.bUnLit, P.Default.bMeshEnviromap);
	}
	else
	{
		P.bBlockPlayers = true;
		P.SetDisplayProperties(STY_Normal, P.default.Texture, P.default.bUnLit, P.Default.bMeshEnviromap);
	}
}
//### END OF ANTI-CAMPER FUNCTIONS

//#########################################################################
//### PLAYER AND RECORD MANAGMENT FUNCTIONS
//#########################################################################
//====================================
// CheckForNewPlayer - Check for new player
// Triggered in: Tick, ModifyPlayer
//====================================
function CheckForNewPlayer()
{
	local Pawn Other;

	if(Level.Game.CurrentID > CurrentID) // At least one new player has joined - sometimes this happens faster than tick
	{
		for( Other=Level.PawnList; Other!=None; Other=Other.NextPawn )
			if(Other.PlayerReplicationInfo.PlayerID == CurrentID)
				break;
		CurrentID++;

		// Make sure it is a player.
		if(Other == none || !Other.bIsPlayer || !Other.IsA('PlayerPawn'))
			return;
		if(Other.PlayerReplicationInfo.bIsSpectator && !Other.PlayerReplicationInfo.bWaitingPlayer)
			InitNewSpec(Other);
		else
			InitNewPlayer(Other);
	}
}

//====================================
// InitNewPlayer - Check for new player
// Triggered in: CheckForNewPlayer
//====================================
function InitNewPlayer(Pawn P)
{
	local int i;
	i=FindFreePISlot();
	PI[i].Player=P;
	PI[i].PlayerID=P.PlayerReplicationInfo.PlayerID;
	PI[i].bHelpSent=False;
	PI[i].BestTime=0;
	if(PI[i].RI != none)
	{
		PI[i].RI.Destroy();
		PI[i].RI = None;
	}
	PI[i].Config = spawn(class'UserConfig',P);
	PI[i].RI = spawn(class'BTPPReplicationInfo',P);
	PI[i].RI.IpToCountry = IpToCountry;
	PI[i].RI.PlayerID = P.PlayerReplicationInfo.PlayerID;
	PI[i].LastPlayerLoc = P.Location;
	Pi[i].RI.JoinTime=Level.TimeSeconds;
}

//====================================
// InitNewPlayer - Check for new player
// Triggered in: Tick
//====================================
function InitNewSpec(Pawn P)
{
	local int i;

	i=FindFreeSISlot();
	SI[i].RI = spawn(class'BTPPReplicationInfo',P);
	SI[i].RI.SetVariance(Level.TimeSeconds);
	SI[i].Spec = P;
}

//====================================
// FindFreePISlot - Find a free place in a PlayerInfo struct
// Triggered in: InitNewPlayer
//====================================
function int FindFreePISlot()
{
	local int i;

	for(i=0;i<32;i++)
	{
		if(PlayerPawn(PI[i].Player) == none)
			return i;
		else if(PlayerPawn(PI[i].Player).Player == none)
				return i;
	}
}

//====================================
// FindFreeSISlot - Find a free place in a SpecInfo struct
// Triggered in: InitNewPlayer
//====================================
function int FindFreeSISlot()
{
	local int i;
	for(i=0;i<32;i++)
		if(SI[i].Spec == none)
			return i;
		else if(PlayerPawn(SI[i].Spec).Player == none)
			return i;
}

//====================================
// FindPlayer - Find a player in the PlayerInfo struct by a Pawn object
// Triggered in: Almost everywhere :P
//====================================
function int FindPlayer(Pawn P)
{
	local int i, ID;
	ID=P.PlayerReplicationInfo.PlayerID;
	for(i=0;i<32;i++)
	{
		if(PI[i].PlayerID == ID)
			return i;
	}
}

//====================================
// FindPlayer - Find a player in the PlayerInfo struct by a PlayerID
// Triggered in: GetItemName
//====================================
function int FindPlayerByID(coerce int ID)
{
	local int i;
	for(i=0;i<32;i++)
		if(PI[i].PlayerID == ID)
			return i;
}

//====================================
// FindPlayer - Find a best time in the PlayerInfo struct and returns its owner index in that struct
// Triggered in: SetBestTime
//====================================
function int GetMapBestPlayerIndex()
{
 	local int i, BestTimeID, BestTime;

 	BestTime = 0;

 	for(i=0;i<32;i++)
	{
 		if(PI[i].RI == none || PI[i].BestTime == 0)
 			continue;
 		if(PI[i].BestTime > BestTime)
 		{
			BestTime = PI[i].BestTime;
 			BestTimeID = i;
 		}
 	}
	if(BestTime == 0 || BestTime == 2000)
		return -1;
	else
		return BestTimeID;
}

//====================================
// GetBestTimeClient, GetBestTimeServer, GetSTFU, CheckIfBoosted, SetNoneFlag - used to access structs data from FlagDisposer, trying to acces it normally would result in "too complex variable error"
// Triggered in: class'FlagDisposer'.Touch()
//====================================
function int GetBestTimeClient(Pawn P) { return PI[FindPlayer(P)].Config.BestTime; }
function int GetBestTimeServer(Pawn P) { return PI[FindPlayer(P)].BestTime; }
function bool GetSTFU(Pawn P) { return PI[FindPlayer(P)].Config.bSTFU; }
function bool CheckIfBoosted(Pawn P) { return PI[FindPlayer(P)].RI.bBoosted; }
function bool SetNoneFlag(Pawn P) { PI[FindPlayer(P)].RI.SetNoneFlag(); }

//====================================
// SetBestTime - Saves a new record, in clientside if needed, in serverside. It also informs other players about new record.
// Triggered in: class'FlagDisposer'.Touch()
//====================================
function SetBestTime(Pawn P, int Time)
{
	local int i, j;
	local string Nick;

	i = FindPlayer(P);

	// save the time clientside
	if(Time > PI[i].Config.BestTime)
		PI[i].Config.AddRecord(Time, CurTimestamp());

	if(Time > PI[i].BestTime)
	{
		PI[i].BestTime = Time;
		PI[i].RI.BestTime = Time;
	}

	j = GetMapBestPlayerIndex();
	if(j==-1)
		return;
	if(PI[j].BestTime > MapBestTime)
	{
		if(PI[j].Player != none)
		{
			Nick = PI[j].Player.PlayerReplicationInfo.PlayerName;
			MapBestTime = PI[j].BestTime;
			MapBestPlayer = Nick;
		}
		// save the time serverside
		if(bSaveRecords)
			RD.AddRecord(LevelName, MapBestTime, MapBestPlayer, CurTimestamp());
		SendEvent("server_record", PI[j].PlayerID, MapBestTime);
		GRI.MapBestTime = FormatScore(MapBestTime);
		GRI.MapBestPlayer = MapBestPlayer;
		for(i=0;i<32;i++)
		{
		 	if(PI[i].RI == none)
		 		continue;
			if(PI[j].Player != PI[i].Player && !PI[i].Config.bSTFU)
			{
    			PlayerPawn(PI[i].Player).ClearProgressMessages();
    			PlayerPawn(PI[i].Player).SetProgressTime(8);
				PlayerPawn(PI[i].Player).SetProgressMessage(MapBestPlayer$" has set up a new map record !!!", 5);
   				PlayerPawn(PI[i].Player).SetProgressColor(GreenColor, 6);
				PlayerPawn(PI[i].Player).SetProgressMessage(FormatScore(MapBestTime), 6);
			}
		}
		for(i=0;i<32;i++)
		{
			if(SI[i].RI == none)
				continue;
   		PlayerPawn(SI[i].Spec).ClearProgressMessages();
   		PlayerPawn(SI[i].Spec).SetProgressTime(8);
			PlayerPawn(SI[i].Spec).SetProgressMessage(MapBestPlayer$" has set up a new map record !!!", 5);
			PlayerPawn(SI[i].Spec).SetProgressColor(GreenColor, 6);
			PlayerPawn(SI[i].Spec).SetProgressMessage(FormatScore(MapBestTime), 6);
		}
	}
}

//====================================
// PlayerCapped - Event on cap, increases some variables.
// Triggered in: class'FlagDisposer'.Touch()
//====================================
function PlayerCapped(Pawn P)
{
	local int i;
	i=FindPlayer(P);
	PI[i].RI.Caps++;
	PI[i].RI.Runs++;
}

//====================================
// MsgAll - Sends message to all players
// Triggered in: Mutate
//====================================
function MsgAll(string Msg)
{
	local int i;

	for(i=0;i<32;i++)
	{
		if(PI[i].RI==none)
			continue;
		if(!PI[i].Config.bSTFU)
			PI[i].Player.ClientMessage(Msg);
	}
}

//====================================
// DisableAllGhost - Disables the ghost status on all players
// Triggered in: Mutate
//====================================
function DisableAllGhost()
{
	local int i;
	for(i=0;i<32;i++)
	{
		if(PI[i].Player == None )
			continue;

		SetGhost(PI[i].Player, false);
	}
}

//====================================
// DisableAllGhost - Enables the ghost status on all players
// Triggered in: Mutate
//====================================
function EnableAllGhost()
{
	local int i;
	for(i=0;i<32;i++)
	{
		if(PI[i].Player == None )
			continue;

		SetGhost(PI[i].Player, true);
	}
}

//====================================
// SendHelp - Sends a help message to a specified Pawn
// Triggered in: Timer, Mutate
//====================================
function SendHelp(Pawn P)
{
	if(PI[FindPlayer(P)].Config.bSTFU)
		return;
	P.ClientMessage("BTPlusPlus v0."$Version$" by [es]Rush*bR");
	P.ClientMessage("Type 'mutate bthelp' in console for more info.");
}

//====================================
// SetAntiBoostOn - Enables AntiBoost for a specified Pawn
// Triggered in: MutatorTeamMessage, Mutate
//====================================
function SetAntiBoostOn(Pawn P, int ID)
{
	P.ClientMessage("You can't be boosted from now!");
	if(!PI[ID].Config.bSet) // will be true only first time, cause bSet=true after the next call
		P.ClientMessage("Command 'mutate boost' or 'say boost' will allow boosting.");
	PI[ID].Config.SetAntiBoost(True);
}

//====================================
// SetAntiBoostOff - Disables AntiBoost for a specified Pawn
// Triggered in: MutatorTeamMessage, Mutate
//====================================
function SetAntiBoostOff(Pawn P, int ID)
{
	P.ClientMessage("You can now be boosted !");
	if(!PI[ID].Config.bSet) // will be true only first time, cause bSet=true after the next call
		P.ClientMessage("Command 'mutate noboost' or 'say noboost' will block boosting");
	PI[ID].Config.SetAntiBoost(False);
}

//====================================
// UpdatePlayerStartTimes - It sets a start time for the timer counting seconds from the last call of this function
// Triggered in: class'FlagDisposer'.Touch
//====================================
function UpdatePlayerStartTimes(Pawn P, optional float Modifier)
{
	local int ID;
	ID=FindPlayer(P);
	PI[ID].RI.SetStartTime(Modifier);
	P.PlayerReplicationInfo.StartTime=Level.TimeSeconds-Modifier;
}

//### END OF PLAYER AND RECORD MANAGMENT FUNCTIONS

//#########################################################################
//### TEXT FUNCTIONS - used all over the place :)
//#########################################################################
//====================================
// ElementsNum - Counts a specified character in a string(and thus elements) and returns the countresult-1;
//====================================
static final function int ElementsNum(string Str, optional string Char)
{
	local int count, pos;

	if(Char=="")
		Char=":"; // this is a default separator
	while(true)
	{
		pos = InStr(Str, Char);
		if(pos == -1)
			break;
		Str=Mid(Str, pos+1);
		count++;
	}
	return count+1;
}

//====================================
// SelElem - Selects an element from a string where elements are separated by a "Char"
//====================================
static final function string SelElem(string Str, int Elem, optional string Char)
{
	local int pos, count;
	if(Char=="")
		Char=":"; // this is a default separator

	while( (Elem--) >1)
	{
		pos=InStr(Str, Char);
		if(pos != -1)
			Str=Mid(Str, pos+1);
		else
			return "";
	}
	pos=InStr(Str, Char);
	if(pos != -1)
    	return Left(Str, pos);
	else
		return Str;
}

//====================================
// SepLeft - Separates a left part of a string with a certain character as a separator
//====================================
static final function string SepLeft(string Input, optional string Char)
{
	local int pos;
	if(Char=="")
		Char=":"; // this is a default separator

	pos = InStr(Input, Char);
	if(pos != -1)
		return Left(Input, pos);
	else
		return "";
}

//====================================
// SepLeft - Separates a right part of a string with a certain character as a separator
//====================================
static final function string SepRight(string Input, optional string Char)
{
	local int pos;
	if(Char=="")
		Char=":"; // this is a default separator

	pos = InStr(Input, Char);
	if(pos != -1)
		return Mid(Input, pos+1);
	else
		return "";
}

//====================================
// DelSpaces - Deletes spaces from an end of a string
//====================================
static final function string DelSpaces(string Input)
{
	local int pos;
	pos = InStr(Input, " ");
	if(pos != -1)
		return Left(Input, pos);
	else
		return Input;
}

//====================================
// FormatScore - Format BunnyTrack score to a readable format - the code is stupid because it has to be like in the original one in BunnyTrack mod to maintain compatibility
// Triggered in: PostBeginPlay, SetBestTime
//====================================
static final function string FormatScore(coerce int Score)
{
	local int secs;
	local string sec;

	if(Score==0 || Score==2000)
		return "-:--";
	else if ( Score >= 1 && Score <= 1999 )
	{
		Score = 2000 - Score;
		secs = int(Score % 60);
		if ( secs < 10 )
                	sec = "0" $string(secs);
            	else
			sec = "" $string(secs);
		return string(Score / 60) $":"$sec;
	}
}

//====================================
// GetLevelName - Returns a level name(file name) in a readable format
// Triggered in: PostBeginPlay
//====================================
function string GetLevelName()
{
	local string Str;
	local int Pos;

	Str=string(Level);
	Pos = InStr(Str, ".");
	if(Pos != -1)
		return Left(Str, Pos);
	else
		return Str;
}
//### END OF TEXT FUNCTIONS


//#########################################################################
//### MUTATOR FUNCTIONS - inherited from class'Mutator'
//#########################################################################
//====================================
// AddMutator = Little security against initializing this script twice.
//====================================
function AddMutator(Mutator M)
{
	if ( M.Class != Class )
		Super.AddMutator(M);
	else if ( M != Self )
		M.Destroy();
}

//====================================
// MutatorTakeDamage - checks for instagib rays trying to boost a player or to kill someone, it prevents or allows it
// Inherited from class'Mutator'
//====================================
function MutatorTakeDamage( out int ActualDamage, Pawn Victim, Pawn InstigatedBy, out Vector HitLocation, out Vector Momentum, name DamageType)
{
	local int VictimID;
	local int BoosterID;

	// only for player versus player and instagib damage type
	if(string(DamageType)=="jolted" && InstigatedBy.IsA('PlayerPawn') && Victim.IsA('PlayerPawn'))
	{
		VictimID=FindPlayer(Victim);
		BoosterID=FindPlayer(InstigatedBy);

		if( ((PI[VictimID].Config.bAntiBoost && bAntiBoost) || (InstigatedBy.PlayerReplicationInfo.Team != Victim.PlayerReplicationInfo.Team) || bBlockBoostForGood) && !bAllowBoost)
		{
			Momentum = Vect(0,0,0);
		}
		else if((RecordsWithoutBoost == 1 || (RecordsWithoutBoost == 2 && !bCooperationMap)) && InstigatedBy.PlayerReplicationInfo.Team == Victim.PlayerReplicationInfo.Team)
		{
			if(!PI[VictimID].RI.bBoosted)
			{
				PI[VictimID].RI.bBoosted = True;
				SendEvent("boost_record_prevent", PI[VictimID].PlayerID, PI[BoosterID].PlayerID);
			}
		}
		else if(InstigatedBy.PlayerReplicationInfo.Team == Victim.PlayerReplicationInfo.Team)
			SendEvent("boost", PI[VictimID].PlayerID, PI[BoosterID].PlayerID);

		if(((Victim.PlayerReplicationInfo.HasFlag == none || bMultiFlags) && bNoKilling && !bAllowKilling) || (InstigatedBy.PlayerReplicationInfo.Team == Victim.PlayerReplicationInfo.Team))
			ActualDamage = 0;
	}

	if ( NextDamageMutator != None )
		NextDamageMutator.MutatorTakeDamage( ActualDamage, Victim, InstigatedBy, HitLocation, Momentum, DamageType );
}

//====================================
// ScoreKill - Decreases points increased by a standard function, updates number of runs and sets start times for the timer
//====================================
function ScoreKill(Pawn Killer, Pawn Other)
{
	local int ID;
	if ( NextMutator != None )
		NextMutator.ScoreKill(Killer, Other);

	if(Killer != None)
		if(Killer.IsA('PlayerPawn'))
			Killer.PlayerReplicationInfo.Score-=1.0;
	ID=FindPlayer(Other);
	PI[ID].RI.Runs++;
	Other.PlayerReplicationInfo.StartTime = Level.TimeSeconds;
	PI[ID].RI.SetStartTime();
}


//====================================
// ModifyPlayer - Checks for new player and sets some variables
//====================================
function ModifyPlayer(Pawn Other)
{
	local int ID;
	if ( NextMutator != None )
		NextMutator.ModifyPlayer(Other);

	if(!Other.IsA('PlayerPawn'))
		return;

	CheckForNewPlayer(); // sometimes modifyplayer is being called faster than a tick where usual new player detection is done thus we have to search for new players also here

	if(bEverybodyGhosts)
		SetGhost(Other, True);

	ID=FindPlayer(Other);

	PI[ID].RI.bBoosted = False;

	Other.bHidden=False; // hidden was set after the cap in class'FlagDisposer' in order to prevent some bugs in showing the player

	// Will cause sending a message to new players.
	if ( Other.PlayerReplicationInfo.Deaths==0 && PI[ID].RI.Runs == 0) /* first spawn */
	{
		settimer(10, false);
		PI[ID].RI.SetStartTime(1);
		Other.PlayerReplicationInfo.StartTime = Level.TimeSeconds-1;
	}
}

//====================================
// MutatorTeamMessage - Allows changing antiboost status also with normal say messages
//====================================
function bool MutatorTeamMessage(Actor Sender, Pawn Receiver, PlayerReplicationInfo PRI, coerce string S, name Type, optional bool bBeep)
{
	local int ID;

	if(Sender.IsA('PlayerPawn'))
	{
		ID = FindPlayer(PlayerPawn(Sender));
		if(S ~= "BOOST" && bAntiBoost && PI[ID].Config.bAntiBoost && !bBlockBoostForGood)
			SetAntiBoostOff(PlayerPawn(Sender), ID);
		else if(S ~= "NOBOOST" && bAntiBoost && !PI[ID].Config.bAntiBoost && !bBlockBoostForGood)
			SetAntiBoostOn(PlayerPawn(Sender), ID);
	}
	if ( NextMessageMutator != None )
		return NextMessageMutator.MutatorTeamMessage( Sender, Receiver, PRI, S, Type, bBeep );
	else
		return true;
}

//====================================
// Mutate - Shows help, allows to change the settings and provides an interface to searching the record database
//====================================
function Mutate(string MutateString, PlayerPawn Sender)
{
	local string CommandString;
	local string ValueString;
	local string TempString;
	local int i, j, ID;

	if ( NextMutator != None )
		NextMutator.Mutate(MutateString, Sender);

	if((!Sender.PlayerReplicationInfo.bIsSpectator || Sender.PlayerReplicationInfo.bWaitingPlayer))
	{
		ID = FindPlayer(Sender);
		switch(Caps(MutateString))
		{
			case "BTSETTINGS":
				Sender.ClientMessage("BT++ Settings:");
				SendSettings(Sender);
				break;
			case "BTHELP":
			case "BT++HELP":
			case "BTPPHELP":
				Sender.ClientMessage("BT++ Client Commands (type directly in console or bind to key):");
				if(bBlockBoostForGood)
					Sender.ClientMessage("Caution! Boosting is forbidden on this server.");
				Sender.ClientMessage("Mutate:");
				if(bSaveRecords)
					Sender.ClientMessage("- records (Search for time records on current server)");
				if(!bBlockBoostForGood)
				{
					Sender.ClientMessage("- boost (Allow others to boost you)");
					Sender.ClientMessage("- noboost (Deny others to boost you)");
				}
				Sender.ClientMessage("- bthud (Display improved HUD)");
				Sender.ClientMessage("- nobthud (Hide improved HUD)");
				Sender.ClientMessage("- btstfu (Mute BT++ messages)");
				Sender.ClientMessage("- btnostfu (Unmute BT++ messages)");
				Sender.ClientMessage("- btsettings (Reveal BT++ configuration)");
				Sender.ClientMessage("- bthelp (Show this help)");
				Sender.ClientMessage("For admins:");
				Sender.ClientMessage("- BTPP (BT++ settings)");
				Sender.ClientMessage("** Settings will be stored in your User.ini");
				if(!bBlockBoostForGood)
					Sender.ClientMessage("** boost and noboost can be just set with SAY instead of MUTATE");
				break;
			case "AB_OFF":
			case "BOOST_ON":
			case "BOOST":
				if(PI[ID].Config.bAntiBoost && bAntiBoost && !bBlockBoostForGood)
					SetAntiBoostOff(Sender, ID);
				break;
			case "AB_ON":
			case "BOOST_OFF":
			case "NOBOOST":
				if(!PI[ID].Config.bAntiBoost && bAntiBoost && !bBlockBoostForGood)
					SetAntiBoostOn(Sender, ID);
				break;
			case "ABHELP":
			case "AB_HELP":
			case "BOOSTHELP":
				SendHelp(Sender);
				break;
			case "NOBTHUD":
				PI[ID].Config.SetBTHud(False);
				break;
			case "BTHUD":
				PI[ID].Config.SetBTHud(True);
				break;
			case "BTSTFU":
				if(PI[ID].Config.bSTFU)
				{
					Sender.ClientMessage("Already set.");
				}
				else
					Sender.ClientMessage("So be it! BTPlusPlus won't say a word to you. 'mutate btnostfu' would make me speak again.");
				PI[ID].Config.SetSTFU(True);
				break;
			case "BTNOSTFU":
				if(!PI[ID].Config.bSTFU)
				{
					Sender.ClientMessage("Already set.");
				}
				else
					Sender.ClientMessage("Yeah, you allowed me to speak again.");
				PI[ID].Config.SetSTFU(False);
				break;
		}
	}
	if(Left(MutateString, 4) ~= "BTPP" && !Sender.bAdmin )
			Sender.ClientMessage("You cannot set BTPlusPlus until you're a serveradmin.");
	else if(Sender.bAdmin)
	{
		if(Left(MutateString, 4) ~= "BTPP" && Len(MutateString)==4)
		{
			Sender.ClientMessage("BTPlusPlus v0."$Version$" - Configuration menu:");
			Sender.ClientMessage("Settings:");
			SendSettings(Sender);
			Sender.ClientMessage("To set something type 'mutate btpp <option> <value>'.");
			Sender.ClientMessage("On/Off command: 'mutate antiboost enabled/disabled'.");
		}
		else if(Left(MutateString, 5) ~= "BTPP ")
		{
			CommandString=Mid(MutateString, 5);
			if(Left(CommandString , 7) ~= "ENABLED" && !bEnabled)
			{
				bEnabled=True;
				Sender.ClientMessage("Applied. So you changed your mind ?");
//				EnableBTPlusPlus();
			}
			else if(Left(CommandString, 8) ~= "DISABLED" && bEnabled)
			{
				bEnabled=False;
				Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
//				DisableBTPlusPlus();
			}
			else if(Left(CommandString, 14)~="BBTSCOREBOARD ")
			{
				ValueString=Mid(CommandString, 14);
				if(ValueString~="TRUE" && !bBTScoreboard)
				{
					bAutoLoadInsta=True;
					Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
				}
				else if(ValueString~="FALSE" && bBTScoreboard)
				{
					bAutoLoadInsta=False;
					Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
				}
			}
			else if(Left(CommandString, 15)~="BAUTOLOADINSTA ")
			{
				ValueString=Mid(CommandString, 15);
				if(ValueString~="TRUE" && !bAutoLoadInsta)
				{
					bAutoLoadInsta=True;
					Sender.ClientMessage("Applied. Instagib will be loaded after the next map start.");
				}
				else if(ValueString~="FALSE" && bAutoLoadInsta)
				{
					bAutoLoadInsta=False;
					Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
				}
			}
			else if(Left(CommandString, 12)~="BMULTIFLAGS ")
			{
				ValueString=Mid(CommandString, 12);
				if(ValueString~="TRUE" && !bMultiFlags)
				{
					bMultiFlags=True;
					SetMultiFlags(True);
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bMultiFlags)
				{
					bMultiFlags=False;
					SetMultiFlags(False);
					Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 17)~="BRESPAWNAFTERCAP ")
			{
				ValueString=Mid(CommandString, 17);
				if(ValueString~="TRUE" && !bRespawnAfterCap)
				{
					bRespawnAfterCap=True;
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bRespawnAfterCap)
				{
					bRespawnAfterCap=False;
					Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 11)~="BANTIBOOST ")
			{
				ValueString=Mid(CommandString, 11);
				if(ValueString~="TRUE" && !bAntiBoost)
				{
					bAntiBoost=True;
					GRI.bShowAntiBoostStatus=True;
					Sender.ClientMessage("Applied.");
					MsgAll("AntiBoost function activated.");
				}
				else if(ValueString~="FALSE" && bAntiBoost)
				{
					bAntiBoost=False;
					GRI.bShowAntiBoostStatus=False;
					Sender.ClientMessage("Applied.");
					MsgAll("AntiBoost function disabled.");
				}
			}
			else if(Left(CommandString, 19)~="BBLOCKBOOSTFORGOOD ")
			{
				ValueString=Mid(CommandString, 19);
				if(ValueString~="TRUE" && !bBlockBoostForGood)
				{
					bBlockBoostForGood = True;
					if(bAntiBoost)
						GRI.bShowAntiBoostStatus=False;
					Sender.ClientMessage("Applied.");
					MsgAll("Boosting is forbidden from now.");
				}
				else if(ValueString~="FALSE" && bBlockBoostForGood)
				{
					bBlockBoostForGood = False;
					if(bAntiBoost)
						GRI.bShowAntiBoostStatus=True;
					Sender.ClientMessage("Applied.");
					MsgAll("Boosting is allowed from now.");
				}
			}
			else if(Left(CommandString, 13)~="ALLOWBOOSTON ")
			{
				ValueString=Mid(CommandString, 13);
				AllowBoostOn=ValueString;
				CheckAllowBoost();
				Sender.ClientMessage("Applied.");
			}
			else if(Left(CommandString, 11)~="BNOKILLING ")
			{
				ValueString=Mid(CommandString, 11);
				if(ValueString~="TRUE" && !bNoKilling)
				{
					bNoKilling=True;
					Sender.ClientMessage("Applied.");
					CheckBlockKilling();
				}
				else if(ValueString~="FALSE" && bNoKilling)
				{
					bNoKilling=False;
					Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 18)~="BGHOSTWHENCAMPING ")
			{
				ValueString=Mid(CommandString, 18);
				if(ValueString~="TRUE" && !bGhostWhenCamping)
				{
					bGhostWhenCamping=True;
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bGhostWhenCamping)
				{
					bGhostWhenCamping=False;
					DisableAllGhost();
					Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 11)~="CAMPRADIUS ")
			{
				ValueString=Mid(CommandString, 11);
				CampRadius=Min(Max(int(ValueString), 0), 1000);
				Sender.ClientMessage("Applied.");
			}
			else if(Left(CommandString, 9)~="CAMPTIME ")
			{
				ValueString=Mid(CommandString, 9);
				CampTime=Min(Max(int(ValueString), 0), 1000);
				Sender.ClientMessage("Applied.");
			}
			else if(Left(CommandString, 17)~="BEVERYBODYGHOSTS ")
			{
				ValueString=Mid(CommandString, 17);
				if(ValueString~="TRUE" && !bEverybodyGhosts)
				{
					bEverybodyGhosts=True;
					EnableAllGhost();
					Sender.ClientMessage("Applied.");
					MsgAll("You have become a ghost.");
				}
				else if(ValueString~="FALSE" && bEverybodyGhosts)
				{
					bEverybodyGhosts=False;
					DisableAllGhost();
					Sender.ClientMessage("Applied.");
					MsgAll("You are not a ghost now.");
				}
			}
			else if(Left(CommandString, 17)~="BFORCEMOVERSKILL ")
			{
				ValueString=Mid(CommandString, 17);
				if(ValueString~="TRUE" && !bForceMoversKill)
				{
					bForceMoversKill=True;
					CheckMovers();
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bForceMoversKill)
				{
					bForceMoversKill=False;
					CheckMovers(True);
					Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 15)~="FORCEMOVERSKILL")
			{
				for(i=0;i<10;i++)
				{
					TempString=Left(CommandString, 15)$"["$string(i)$"]";
					if(Left(CommandString, 18)~=TempString)
					{
						ValueString=Mid(CommandString, 19);
						for(j=0;j<10 && ValueString!="";j++)
						{
							if(ForceMoversKill[j]==ValueString)
								break;
						}

						if(ForceMoversKill[j]==ValueString && ValueString!="")
							Sender.ClientMessage("Error. ForceMoversKill["$string(j)$"] has the same value.");
						else
						{
							if(bForceMoversKill)
								CheckMovers();
							else
								CheckMovers(True);
							Sender.ClientMessage("Applied.");
							ForceMoversKill[i]=ValueString;
							break;
						}
					}
				}
			}
			else if(Left(CommandString, 15)~="ALLOWKILLINGON ")
			{
				ValueString=Mid(CommandString, 15);
				AllowKillingOn=ValueString;
				CheckBlockKilling();
				Sender.ClientMessage("Applied.");
			}
			else if(Left(CommandString, 12)~="BSAVERECORDS ")
			{
				ValueString=Mid(CommandString, 12);
				if(ValueString~="TRUE" && !bSaveRecords)
				{
					SetSaveRecords(True);
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bSaveRecords)
				{
				 	SetSaveRecords(False);
				 	Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 14)~="BNOCAPSUICIDE ")
			{
				ValueString=Mid(CommandString, 14);
				if(ValueString~="TRUE")
				{
					SetSaveRecords(True);
					Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE")
				{
				 	SetSaveRecords(False);
				 	Sender.ClientMessage("Applied.");
				}
			}
			else if(Left(CommandString, 20)~="RECORDSWITHOUTBOOST ")
			{
				ValueString=Mid(CommandString, 20);
				switch(int(ValueString))
				{
					case 0:
					case 1:
					case 2:
						SetRecordsWithoutBoost(int(ValueString));
						Sender.ClientMessage("Applied.");
						break;
					default:
						Sender.ClientMessage("Must be 0, 1 or 2.");
				}
			}
			else if(Left(CommandString, 21)~="BDISABLEINTOURNAMENT ")
			{
				ValueString=Mid(CommandString, 21);
				if(ValueString~="TRUE" && !bDisableInTournament)
				{
					bDisableInTournament=True;
					if(class<DeathMatchPlus>(Level.Game.Class).Default.bTournament)
					{					;
						Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
					}
					else
						Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bDisableInTournament)
				{
					bDisableInTournament=False;
					Sender.ClientMessage("Applied.");
					if(class<DeathMatchPlus>(Level.Game.Class).Default.bTournament && ((Left(string(Level), 3)=="BT-" || Left(string(Level), 6)=="CTF-BT-") || !bDisableInNonBTMaps ))
					{
						Sender.ClientMessage("Applied. So you've changed your mind ?");
					}
				}
			}
			else if(Left(CommandString, 20)~="BDISABLEINNONBTMAPS ")
			{
				ValueString=Mid(CommandString, 20);
				if(ValueString~="TRUE" && !bDisableInNonBTMaps)
				{
					bDisableInNonBTMaps=True;
					if(Left(string(Level), 3)!="BT-" && Left(string(Level), 6)!="CTF-BT-")
					{
						Sender.ClientMessage("Applied. However you need to change the map for it to take effect.");
					}
					else
						Sender.ClientMessage("Applied.");
				}
				else if(ValueString~="FALSE" && bDisableInNonBTMaps)
				{
					bDisableInNonBTMaps=False;
					Sender.ClientMessage("Applied.");
					if(Left(string(Level), 3)!="BT-" && Left(string(Level), 6)!="CTF-BT-" && (!class<DeathMatchPlus>(Level.Game.Class).Default.bTournament || !bDisableInTournament))
					{
						Sender.ClientMessage("Applied. So you've changed your mind ?");
					}
				}
			}
			else if(Left(CommandString, 11)~="BOARDLABEL ")
			{
				ValueString=Mid(CommandString, 11);
				BoardLabel=ValueString;
				Sender.ClientMessage("Applied.");
			}
			else if(Left(CommandString, 4)~="GET ")
			{
				ValueString=Caps(Mid(CommandString, 4));
				switch(ValueString) {
				case "BBTSCOREBOARD":
					Sender.ClientMessage(string(bBTScoreboard)); break;
				case "BANTIBOOST":
					Sender.ClientMessage(string(bAntiBoost)); break;
				case "BBLOCKBOOSTFORGOOD":
					Sender.ClientMessage(string(bBlockBoostForGood));
				case "ALLOWBOOSTON":
					Sender.ClientMessage(AllowBoostOn);
				case "BNOKILLING":
					Sender.ClientMessage(string(bNoKilling)); break;
				case "ALLOWKILLINGON":
					Sender.ClientMessage(AllowKillingOn); break;
				case "BAUTOLOADINSTA":
					Sender.ClientMessage(string(bAutoLoadInsta)); break;
				case "BGHOSTWHENCAMPING":
					Sender.ClientMessage(string(bGhostWhenCamping)); break;
				case "CAMPTIME":
					Sender.ClientMessage(string(CampTime)); break;
				case "CAMPRADIUS":
					Sender.ClientMessage(string(CampRadius)); break;
				case "BFORCEMOVERSKILL":
					Sender.ClientMessage(string(bForceMoversKill)); break;
				case "FORCEMOVERSKILL[0]":
					Sender.ClientMessage(ForceMoversKill[0]); break;
				case "FORCEMOVERSKILL[1]":
					Sender.ClientMessage(ForceMoversKill[1]); break;
				case "FORCEMOVERSKILL[2]":
					Sender.ClientMessage(ForceMoversKill[2]); break;
				case "FORCEMOVERSKILL[3]":
					Sender.ClientMessage(ForceMoversKill[3]); break;
				case "FORCEMOVERSKILL[4]":
					Sender.ClientMessage(ForceMoversKill[4]); break;
				case "FORCEMOVERSKILL[5]":
					Sender.ClientMessage(ForceMoversKill[5]); break;
				case "FORCEMOVERSKILL[6]":
					Sender.ClientMessage(ForceMoversKill[6]); break;
				case "FORCEMOVERSKILL[7]":
					Sender.ClientMessage(ForceMoversKill[7]); break;
				case "FORCEMOVERSKILL[8]":
					Sender.ClientMessage(ForceMoversKill[8]); break;
				case "FORCEMOVERSKILL[9]":
					Sender.ClientMessage(ForceMoversKill[9]); break;
				case "BSAVERECORDS":
					Sender.ClientMessage(string(bSaveRecords)); break;
				case "BNOCAPSUICIDE":
					Sender.ClientMessage(string(bNoCapSuicide)); break;
				case "BDISABLEINTOURNAMENT":
					Sender.ClientMessage(string(bDisableInTournament)); break;
				case "BDISABLEINNONBTMAPS":
					Sender.ClientMessage(string(bDisableInNonBTMaps)); break;
				case "BOARDLABEL":
					Sender.ClientMessage(BoardLabel); break;
				}
			}
			SaveConfig();
		}
	}
	if(Left(MutateString, 8) ~=  "RECORDS " && bSaveRecords)
	{
		CommandString=Mid(MutateString, 8);
		if(Left(CommandString, 4) ~= "MAP ")
		{
			ValueString=Mid(CommandString, 4);
			if(Len(ValueString) > 1)
				RD.FindByMap(Sender, ValueString);
			else
				Sender.ClientMessage("Sorry, the specified string is too short.");
		}
		else if(Left(CommandString, 3) ~= "MAP")
		{
			Sender.ClientMessage("Searches database for map records by map name.");
			Sender.ClientMessage("Use command `mutate records map <mapname>`");
			Sender.ClientMessage("   Note: It will find all records containing the specified string.");
		}
		else if(Left(CommandString, 7) ~= "PLAYER ")
		{
			ValueString=Mid(CommandString, 7);
			if(Len(ValueString) > 1)
				RD.FindByPlayer(Sender, ValueString);
			else
				Sender.ClientMessage("Sorry, the specified string is too short.");
		}
		else if(Left(CommandString, 6) ~= "PLAYER")
		{
			Sender.ClientMessage("Searches database for map records by player name.");
			Sender.ClientMessage("Use command `mutate records map <player>`");
			Sender.ClientMessage("   Note: It will find all records containing the specified string.");
		}
		else
		{
			Sender.ClientMessage("You can do database searches for map records here.");
			Sender.ClientMessage("It can be done in two ways:");
			Sender.ClientMessage("A) By Map - command `mutate records map <mapname>`");
			Sender.ClientMessage("B) By Player - command `mutate records player <player>`");
			Sender.ClientMessage("   Note: It will find all records containing the specified string.");
		}
	}
	else if(Left(MutateString, 7) ~=  "RECORDS" && bSaveRecords)
	{
		Sender.ClientMessage("You can do database searches for map records here.");
		Sender.ClientMessage("It can be done in two ways:");
		Sender.ClientMessage("A) By Map - command `mutate records map <mapname>`");
		Sender.ClientMessage("B) By Player - command `mutate records player <player>`");
		Sender.ClientMessage("   Note: It will find all records containing the specified string.");
	}
}

//====================================
// SendSettings - Sends current server settings - helper for Mutate
// Triggered in: Mutate
//====================================
function SendSettings(Pawn Sender)
{
	local int i;
	Sender.ClientMessage("  bBTScoreboard "$string(bBTScoreboard));
	Sender.ClientMessage("  bAutoLoadInsta "$string(bAutoLoadInsta));
	Sender.ClientMessage("  bMultiFlags "$string(bMultiFlags));
	Sender.ClientMessage("  bRespawnAfterCap "$string(bRespawnAfterCap));
	Sender.ClientMessage("  bAntiBoost "$string(bAntiBoost));
	Sender.ClientMessage("  bBlockBoostForGood "$string(bBlockBoostForGood));
	Sender.ClientMessage("  AllowBoostOn "$AllowBoostOn);
	Sender.ClientMessage("  bNoKilling "$string(bNoKilling));
	Sender.ClientMessage("  AllowKillingOn "$AllowKillingOn);
	Sender.ClientMessage("  bGhostWhenCamping "$string(bGhostWhenCamping));
	Sender.ClientMessage("  CampTime "$string(CampTime));
	Sender.ClientMessage("  CampRadius "$string(CampRadius));
	Sender.ClientMessage("  bEverybodyGhosts "$string(bEverybodyGhosts));
	Sender.ClientMessage("  bForceMoversKill "$string(bForceMoversKill));
	for(i=0;i<10;i++)
		Sender.ClientMessage("  ForceMoversKill["$string(i)$"] "$ForceMoversKill[i]);
	Sender.ClientMessage("  bSaveRecords "$string(bSaveRecords));
	Sender.ClientMessage("  bNoCapSuicide "$string(bNoCapSuicide));
	Sender.ClientMessage("  RecordsWithoutBoost "$string(RecordsWithoutBoost));
	Sender.ClientMessage("  bDisableInTournament "$string(bDisableInTournament));
	Sender.ClientMessage("  bDisableInNonBTMaps "$string(bDisableInNonBTMaps));
	Sender.ClientMessage("  BoardLabel "$BoardLabel);
}
// END OF MUTATOR FUNCTIONS

//====================================
// CheckClassForZP - Checks if a string(which ought to be derived from a class) contains any ZeroPing names
// Triggered in: RemoveActorsForInsta
//====================================
static final function bool CheckClassForZP(coerce string StrClass)
{
		//check for all zp variants
	if (InStr(StrClass,"ZPPure") != -1
	|| InStr(StrClass,"ZeroPing") != -1
	|| InStr(StrClass,"ZPServer") != -1
	|| InStr(StrClass,"ZPBasic") != -1
	|| InStr(StrClass,"ZPColor") != -1
	|| InStr(StrClass,"InstaGibDM") != -1
	|| InStr(StrClass,"ZPDualColor") != -1)
		return true;
	return false;
}

//====================================
// RemoveActorsForInsta - Removes any instagib actors, we have our own one
// Triggered in: PreBeginPlay
//====================================
function RemoveActorsForInsta()
{
	local Inventory I;
	local Actor A;
	local Mutator M, Temp;

	M = Level.Game.BaseMutator;

	while (M.NextMutator != None)
	{
		//check for all zp variants
		if (CheckClassForZP(M.NextMutator.Class))
		{
			Temp=M.NextMutator.NextMutator;
			M.NextMutator.Destroy();
			M.NextMutator = Temp;
			break;
		}
		else
			M = M.NextMutator;
	}

	foreach AllActors(class'Actor', A)
	{
	 	if(CheckClassForZP(A.class))
	 	 	A.Destroy();
	}
}



//#########################################################################
//### FUNCTIONS TO SWITCH BTPLUSPLUS FEATURES ON/OFF
//#########################################################################
//====================================
// SetSaveRecords - sets whether BTPlusPlus should save records in the ini file or not
// Triggered in: Mutate
//====================================
function SetSaveRecords(bool Status)
{
	local int temp2;
	local string temp;

	GRI.bSaveRecords = Status;

	if(Status)
	{
		bSaveRecords = True;
		temp = RD.CheckRecord(LevelName);
		temp2 = int(SepLeft(temp, chr(9)));
		if(temp2 > 0 && temp2 < 2000 && temp2 > MapBestTime)
		{
			MapBestTime = temp2;
			MapBestPlayer = DelSpaces(SepRight(temp, chr(9)));
		}
		else
			RD.AddRecord(LevelName, MapBestTime, MapBestPlayer);
	}
	else
		bSaveRecords = false;
}

//====================================
// SetSaveRecords - sets whether a copy of the flag should be given to the player taking it
// Triggered in: Mutate, Timer
//====================================
function SetMultiFlags(bool Status)
{
	local CTFFlag TmpFlag;

	Foreach AllActors(Class'CTFFlag',TmpFlag)
	{
		if(TmpFlag.Team>1 || TmpFlag.IsA('FlagDisposer'))
			continue;
		TmpFlag.SendHome();
		TmpFlag.SetCollision(!Status, false, false);
	}
}

//====================================
// SetRecordsWithoutBoost - sets whether records should not be saved if a scorer was boosted in the way
// Triggered in: Mutate
//====================================
function SetRecordsWithoutBoost(int Status)
{
	local int i;

	RecordsWithoutBoost = Status;

	if(Status == 0 || (Status == 2 && bCooperationMap))
	{
		for(i=0;i<32;i++)
		{
			if(PI[i].RI==none)
				continue;
			PI[i].RI.bBoosted = False;
		}
	}
}
//### END OF FUNCTIONS TO SWITCH BTPLUSPLUS FEATURES ON/OFF

//#########################################################################
//### OTHER FUNCTIONS - couldn't find a place for them(yet)
//#########################################################################
//====================================
// CurTimestamp - returns current timestamp/unixtime
//====================================
function int CurTimestamp()
{
	return timestamp(Level.Year, Level.Month, Level.Day, Level.Hour, Level.Minute, Level.Second);
}

//====================================
// timestamp - returns timestamp/unixdate for a specified date
//====================================
static final function int timestamp(int year, int mon, int day, int hour, int min, int sec)
{
	/*
		Origin of the algorithm below:
			Linux Kernel <time.h>
	*/
	mon -= 2;
	if (mon <= 0) {	/* 1..12 -> 11,12,1..10 */
		mon += 12;	/* Puts Feb last since it has leap day */
		year -= 1;
	}
	return (((
	    (year/4 - year/100 + year/400 + 367*mon/12 + day) +
	      year*365 - 719499
	    )*24 + hour /* now have hours */
	   )*60 + min  /* now have minutes */
	  )*60 + sec; /* finally seconds */
}

//====================================
// timestamp - just in case of destuction, it's better to clean up the stuff and unlink self
// Inherited from class'Actor'
//====================================
function Destroyed()
{
	local Mutator M;

	if ( Level.Game != None ) {
        if ( Level.Game.BaseMutator == Self )
            Level.Game.BaseMutator = NextMutator;
        if ( Level.Game.DamageMutator == Self )
            Level.Game.DamageMutator = NextDamageMutator;
        if ( Level.Game.MessageMutator == Self )
            Level.Game.MessageMutator = NextMessageMutator;
    }
    ForEach AllActors(Class'Engine.Mutator', M) {
        if ( M.NextMutator == Self )
            M.NextMutator = NextMutator;
        if ( M.NextDamageMutator == Self )
            M.NextDamageMutator = NextDamageMutator;
        if ( M.NextMessageMutator == Self )
            M.NextMessageMutator = NextMessageMutator;
    }
}

//====================================
// GetItemName - provides various information to actors not linked with BT++ directly
// Triggered in: class'SuperShockRifleBT'.ProcessTraceHit and any other actor talking with BT++
// Inherited from class'Actor'
//====================================
function string GetItemName(string Input)
{
	local int temp, PlayerID;
	local string retstr;

	if(Left(Input, 4) ~= "get ")
	{
		Input=Mid(Input, 4);
		PlayerID=int(SelElem(Input, 2, " "));
		switch(SelElem(Caps(Input), 1, " "))
		{
			case "CAPS":
				return string(PI[FindPlayerByID(PlayerID)].RI.Caps);
			case "RUNS":
				return string(PI[FindPlayerByID(PlayerID)].RI.Runs);
			case "EFF":
				temp=FindPlayerByID(PlayerID);
				if(PI[temp].RI.Runs > 0)
					return string(int(float(PI[temp].RI.Caps)/float(PI[temp].RI.Runs)*100.0));
				else
					return "0";
			case "BOOSTED":
				return string(PI[FindPlayerByID(PlayerID)].RI.bBoosted);
			case "ABSTATUS":
				if((PI[FindPlayerByID(Input)].Config.bAntiBoost || bBlockBoostForGood) && bAntiBoost)
					return "1";
				else
					return "0";
			case "BNOKILLING":
				if(bNoKilling)
					return "1";
				else
					return "0";
			case "SHOCKRIFLEINFO": // information needed by the class'SuperShockRifleBT'
				if((PI[FindPlayerByID(Input)].Config.bAntiBoost || bBlockBoostForGood) && bAntiBoost)
					retstr="1";
				else
					retstr="0";
				if(bNoKilling && !bAllowKilling)
					retstr=retstr$"1";
				else
					retstr=retstr$"0";
				return retstr;
			default:
				return "error";
		}
	}
}

event Touch(Actor A)
{
	local int i;

	if(EventHandlersCount == 10)
		A.GetItemName("-1,event handlers number exceeded");

	for(i=0;i<EventHandlersCount+1;i++)
	{
		if(EventHandlers[i] == None)
		{
			EventHandlers[i] = A;
			A.GetItemName("0,registration successful");
			EventHandlersCount++;
		}
	}
}

//====================================
// SendEvent - Sends custom events to all registered actors
//====================================
function SendEvent(string EventName, optional coerce string Arg1, optional coerce string Arg2, optional coerce string Arg3, optional coerce string Arg4)
{
	local Actor A;
	local int i;
	local string Event;

	if (Level.Game.LocalLog != None)
		Level.Game.LocalLog.LogSpecialEvent(EventName, Arg1, Arg2, Arg3, Arg4);

	Event=EventName;
	if(Arg1 != "")
		Event=Event$chr(9)$Arg1;
	if(Arg2 != "")
		Event=Event$chr(9)$Arg2;
	if(Arg3 != "")
		Event=Event$chr(9)$Arg3;
	if(Arg4 != "")
		Event=Event$chr(9)$Arg4;

	for(i=0;i<EventHandlersCount+1;i++)
	{
		if(EventHandlers[i] != None)
			EventHandlers[i].GetItemName(Event);
	}
}
//### END OF OTHER FUNCTIONS

defaultproperties
{
    Version="97"
    bEnabled=True
    bBTScoreboard=True
    bAutoLoadInsta=True
    bMultiFlags=True
    bRespawnAfterCap=True
    bAntiBoost=True
    bBlockBoostForGood=False
    bNoKilling=True
    AllowKillingOn="BT-Colors,BT-1point4megs,BT-i4games,BT-Abomination,BT-Allied"
    bGhostWhenCamping=True
    CampTime=10
    CampRadius=200
    bEverybodyGhosts=False
    bForceMoversKill=True
    ForceMoversKill(0)="Mover0,Mover1,Mover2,Mover5:BT-Maverick"
    bSaveRecords=True
    bNoCapSuicide=True;
    RecordsWithoutBoost=2
    bDisableInTournament=True
    bDisableInNonBTMaps=True
    CountryFlagsPackage=CountryFlags2
    BoardLabel="Bunny Track (BT++)"
    MapBestTime=0
    MapBestPlayer="N/A"
    bAlwaysTick=True
    GreenColor=(G=255)
    RedColor=(R=255)'
    Tag='BTPlusPlus'
}
